/*
  ==============================================================================

    SonicCrypt Flux Gate
    Copyright (C) 2025 Sebastian Snkler

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

  ==============================================================================
*/

#include "PluginProcessor.h"
#include "PluginEditor.h"

SonicCryptFluxGateAudioProcessorEditor::SonicCryptFluxGateAudioProcessorEditor(SonicCryptFluxGateAudioProcessor & p)
    : AudioProcessorEditor(&p), audioProcessor(p)
{
    setResizable(true, true);
    setResizeLimits(800, 600, 1920, 1080);
    setSize(audioProcessor.lastUIWidth, audioProcessor.lastUIHeight);

    // --- SIDEBAR ---
    logoImage = juce::ImageCache::getFromMemory(BinaryData::logo_png, BinaryData::logo_pngSize);
    if (logoImage.isValid()) {
        logoButton.setImages(false, true, true, logoImage, 1.0f, juce::Colours::transparentBlack, logoImage, 1.0f, juce::Colours::white.withAlpha(0.2f), logoImage, 1.0f, juce::Colours::transparentBlack);
    }
    else {
        logoButton.setButtonText("SONIC\nCRYPT");
        logoButton.setColour(juce::TextButton::textColourOffId, juce::Colours::white.withAlpha(0.8f));
    }
    logoButton.setColour(juce::TextButton::buttonColourId, juce::Colours::transparentBlack);
    logoButton.onClick = [] { juce::URL("https://soniccrypt.org").launchInDefaultBrowser(); };
    addAndMakeVisible(logoButton);

    copyrightBtn.setButtonText(juce::CharPointer_UTF8("\xc2\xa9 SonicCrypt"));
    copyrightBtn.setColour(juce::TextButton::buttonColourId, juce::Colours::transparentBlack);
    copyrightBtn.setColour(juce::TextButton::textColourOffId, juce::Colours::grey.withAlpha(0.6f));
    copyrightBtn.onClick = [this] { aboutOverlay.setVisible(true); aboutOverlay.toFront(true); };
    addAndMakeVisible(copyrightBtn);

    // --- PRESETS ---
    addAndMakeVisible(presetGroup);
    presetGroup.setColour(juce::GroupComponent::outlineColourId, juce::Colours::darkgrey);
    presetGroup.setColour(juce::GroupComponent::textColourId, juce::Colours::white);
    addAndMakeVisible(presetListBox);
    presetListBox.setRowHeight(25);
    presetListBox.setColour(juce::ListBox::backgroundColourId, juce::Colours::black.withAlpha(0.3f));
    presetListModel = std::make_unique<PresetListComponent>(presetListBox, [this](juce::File file) { audioProcessor.loadPresetFile(file); });
    presetListBox.setModel(presetListModel.get());
    presetListModel->refresh(audioProcessor.getPresetFolder());

    addAndMakeVisible(savePresetButton); savePresetButton.addListener(this);
    savePresetButton.setColour(juce::TextButton::buttonColourId, juce::Colour::fromString("FF00BFFF"));
    savePresetButton.setColour(juce::TextButton::textColourOffId, juce::Colours::black);
    addAndMakeVisible(refreshButton); refreshButton.addListener(this);
    refreshButton.setColour(juce::TextButton::buttonColourId, juce::Colours::darkgrey);

    // --- VISUALIZER ---
    addAndMakeVisible(visualizer);
    visualizer.pattern = audioProcessor.currentPattern;

    // UPDATE: Neuer Callback mit (Index, Level)
    visualizer.onStepLevelChanged = [this](int stepIndex, float level) {
        audioProcessor.setPatternStepLevel(stepIndex, level);
        visualizer.pattern = audioProcessor.currentPattern;
        visualizer.repaint();
        };

    // --- KNOBS (3x3 GRID) ---
    // Reihe 1
    setupKnob(stepsS, stepsL, stepsAtt, "steps", "STEPS");
    setupKnob(pulsesS, pulsesL, pulsesAtt, "pulses", "DENSITY");
    setupKnob(rotS, rotL, rotAtt, "rotation", "OFFSET");

    // Reihe 2
    setupKnob(swingS, swingL, swingAtt, "swing", "SWING");
    setupKnob(glitchS, glitchL, glitchAtt, "glitch", "CHAOS");
    setupKnob(rateS, rateL, rateAtt, "rate", "RATE");

    // Custom Text fr Rate Slider
    rateS.textFromValueFunction = [](double value) {
        int v = (int)value;
        if (v == 0) return "1/4";
        if (v == 1) return "1/8";
        if (v == 2) return "1/16";
        return "1/32";
        };
    rateS.setNumDecimalPlacesToDisplay(0);

    // Reihe 3
    setupKnob(smoothS, smoothL, smoothAtt, "smooth", "SMOOTH");
    setupKnob(mixS, mixL, mixAtt, "mix", "MIX");
    setupKnob(widthS, widthL, widthAtt, "width", "STEREO");

    // --- FLUX BUTTON ---
    addAndMakeVisible(fluxButton);
    fluxButton.setButtonText("FLUX");
    fluxButton.setColour(juce::TextButton::buttonColourId, juce::Colour::fromString("FF1E1E1E").brighter(0.1f));
    fluxButton.setColour(juce::TextButton::textColourOffId, juce::Colour::fromString("FF00BFFF"));

    fluxButton.onClick = [this] {
        // Triggered Randomization
        audioProcessor.triggerFluxRandomization();
        visualizer.pattern = audioProcessor.currentPattern;
        visualizer.repaint();
        };

    addAndMakeVisible(aboutOverlay); aboutOverlay.setVisible(false);

    // Timer starten
    startTimerHz(60);
    constructorFinished = true;
}

SonicCryptFluxGateAudioProcessorEditor::~SonicCryptFluxGateAudioProcessorEditor() { stopTimer(); }

void SonicCryptFluxGateAudioProcessorEditor::setupKnob(juce::Slider& s, juce::Label& l, std::unique_ptr<SliderAtt>& att, juce::String paramId, juce::String name) {
    addAndMakeVisible(s); addAndMakeVisible(l);
    s.setSliderStyle(juce::Slider::RotaryHorizontalVerticalDrag);
    s.setTextBoxStyle(juce::Slider::NoTextBox, false, 0, 0);
    s.setColour(juce::Slider::rotarySliderFillColourId, juce::Colour::fromString("FF00BFFF"));
    s.setColour(juce::Slider::thumbColourId, juce::Colours::white);
    l.setText(name, juce::dontSendNotification);
    l.setJustificationType(juce::Justification::centred);
    l.setFont(12.0f);
    att = std::make_unique<SliderAtt>(audioProcessor.getAPVTS(), paramId, s);
}

void SonicCryptFluxGateAudioProcessorEditor::paint(juce::Graphics& g)
{
    g.fillAll(juce::Colour::fromString("FF121212"));
    float s = (float)getWidth() / BASE_WIDTH;
    int sidebarW = 230 * s;

    // Sidebar BG
    g.setColour(juce::Colour::fromString("FF1E1E1E"));
    g.fillRoundedRectangle(10.0f * s, 10.0f * s, (float)sidebarW, (float)getHeight() - 20.0f * s, 5.0f);

    // Main Area BG
    float mainX = 10.0f * s + sidebarW + 10.0f * s;
    float mainW = getWidth() - mainX - 10.0f * s;
    g.fillRoundedRectangle(mainX, 10.0f * s, mainW, (float)getHeight() - 20.0f * s, 5.0f);
}

void SonicCryptFluxGateAudioProcessorEditor::resized()
{
    if (constructorFinished) {
        if (getWidth() > 100 && getHeight() > 100) {
            audioProcessor.lastUIWidth = getWidth();
            audioProcessor.lastUIHeight = getHeight();
        }
    }

    float s = (float)getWidth() / BASE_WIDTH;
    int margin = 10 * s;
    int sidebarW = 230 * s;

    // --- SIDEBAR ---
    logoButton.setBounds(margin + 5 * s, margin + 5 * s, sidebarW - 10 * s, 60 * s);
    copyrightBtn.setBounds(margin + 5 * s, margin + 70 * s, sidebarW - 10 * s, 20 * s);

    int presetsY = 110 * s;
    presetGroup.setBounds(margin, presetsY, sidebarW, getHeight() - presetsY - 60 * s);
    presetListBox.setBounds(margin + 5 * s, presetsY + 20 * s, sidebarW - 10 * s, presetGroup.getHeight() - 30 * s);

    int btnY = getHeight() - 50 * s;
    int btnW = (sidebarW - 10 * s) / 2;
    savePresetButton.setBounds(margin + 5 * s, btnY, btnW - 2 * s, 30 * s);
    refreshButton.setBounds(margin + 5 * s + btnW + 2 * s, btnY, btnW - 2 * s, 30 * s);

    // --- MAIN AREA ---
    auto mainArea = getLocalBounds();
    mainArea.removeFromLeft(margin + sidebarW + margin);
    mainArea.removeFromRight(margin);
    mainArea.removeFromTop(margin);
    mainArea.removeFromBottom(margin);

    // 1. FLUX BUTTON (Top)
    auto topArea = mainArea.removeFromTop(60 * s);
    fluxButton.setBounds(topArea.withSizeKeepingCentre(140 * s, 40 * s));

    // 2. CONTROLS (3x3 Grid)
    float knobSize = 65 * s;
    float gap = 20 * s;
    // Hhe = 3 Reihen Knob + Label + bisschen Platz
    float controlsHeight = (knobSize + 25 * s) * 3 + 20 * s;

    auto controlsArea = mainArea.removeFromBottom(controlsHeight);

    float cx = controlsArea.getCentreX();
    float startY = controlsArea.getY();
    float sx = cx - (3 * knobSize + 2 * gap) / 2;

    float rowH = knobSize + 30 * s; // Abstand zwischen Reihen

    // Helper Lambda fr Positionierung
    auto placeRow = [&](juce::Slider& s1, juce::Label& l1,
        juce::Slider& s2, juce::Label& l2,
        juce::Slider& s3, juce::Label& l3,
        float yPos)
        {
            s1.setBounds(sx, yPos, knobSize, knobSize);
            l1.setBounds(sx, yPos - 15 * s, knobSize, 15 * s);

            s2.setBounds(sx + knobSize + gap, yPos, knobSize, knobSize);
            l2.setBounds(sx + knobSize + gap, yPos - 15 * s, knobSize, 15 * s);

            s3.setBounds(sx + 2 * (knobSize + gap), yPos, knobSize, knobSize);
            l3.setBounds(sx + 2 * (knobSize + gap), yPos - 15 * s, knobSize, 15 * s);
        };

    // ROW 1: Generation
    placeRow(stepsS, stepsL, pulsesS, pulsesL, rotS, rotL, startY);

    // ROW 2: Time & Chaos
    placeRow(swingS, swingL, glitchS, glitchL, rateS, rateL, startY + rowH);

    // ROW 3: Output
    placeRow(smoothS, smoothL, mixS, mixL, widthS, widthL, startY + rowH * 2);

    // 3. VISUALIZER
    mainArea.removeFromBottom(20 * s);
    mainArea.removeFromTop(10 * s);
    visualizer.setBounds(mainArea.reduced(10 * s, 10 * s));

    aboutOverlay.setBounds(getLocalBounds());
}

void SonicCryptFluxGateAudioProcessorEditor::timerCallback() {
    visualizer.pattern = audioProcessor.currentPattern;
    visualizer.currentStep = audioProcessor.currentStepIndex;
    visualizer.repaint();
}

void SonicCryptFluxGateAudioProcessorEditor::buttonClicked(juce::Button* button) {
    if (button == &savePresetButton) {
        auto d = audioProcessor.getPresetFolder();
        fileChooser = std::make_shared<juce::FileChooser>("Save Preset", d, "*.scp");
        fileChooser->launchAsync(juce::FileBrowserComponent::saveMode | juce::FileBrowserComponent::canSelectFiles,
            [this](const juce::FileChooser& fc) {
                auto f = fc.getResult();
                if (f != juce::File{}) {
                    if (!f.hasFileExtension("scp")) f = f.withFileExtension("scp");
                    audioProcessor.savePreset(f.getFileNameWithoutExtension());
                    presetListModel->refresh(audioProcessor.getPresetFolder());
                }
            });
    }
    else if (button == &refreshButton) {
        presetListModel->refresh(audioProcessor.getPresetFolder());
    }
}